// **************************************************
//
//       git.rev = 234
//  git.revision = fdd4980be270473bdd7e8206afeda65ab6e4c3a4
//         stage = ES04
//
// ***************************************************
package MusicLandscape.util.formatters;

import MusicLandscape.entities.Track;
import MusicLandscape.util.MyFormatter;

/**
 * This class represents the concept of short formatting of a track containing
 * only some information.
 *
 * @author TeM, JS
 * @version 234
 * @Stage ES04
 * @ProgrammingProblem.Category abstract class
 * @ProgrammingProblem.Introduced ExerciseSheet04
 */
public class ShortTrackFormatter implements MyFormatter<Track> {
    /**
     * Get the headers for the table as a single string.<br>
     * <p>
     * Contains the names for all columns separated by the correct number of
     * blanks.
     *
     * @return the header string.
     */
    @Override
    public String header() {
        return String.format("%-10.10s (%s:%s)", "Title", "min", "sec");
    }

    /**
     * top separator consists of dashes (-) only. It is exactly as wide as the header.
     *
     * @return the separator.
     */
    @Override
    public String topSeparator() {
        return "-".repeat(header().length());
    }

    /**
     * Creates a short format of a track.
     * <p>
     * The short representation of a track is
     * <pre>
     * "title (duration)"
     * </pre>
     * <p>
     * (without quotes).<br>
     * Title is exactly ten characters wide with leading blanks (if any). Duration is in the format minutes:seconds,
     * both at least two digits wide with leading zeros.
     */
    @Override
    public String format(Track t) {
        return String.format("%-10.10s (%02d:%02d)", t.getTitle(), t.getDuration() / 60, t.getDuration() % 60);
    }

    /**
     * the string representation of this formatter is
     * <pre>
     * "short format [Title (min:sec)]" (without quotes)
     * </pre>
     *
     * @return the string representation
     */
    public String toString() {
        return "short format [Title (min:sec)]";
    }
}
